//	AUTHOR:			Maria Karmeliuk, Martin Kocher, Georg Schmidt
//	CREATED:		25/07/2022
//	MODIFIED: 		29/07/2022, by Georg Schmidt
//	DESCRIPTION:	This do-file replicates all results and figures in manuscript #EXEX-D-21-00039R1
//	INSTALL:		To be installed: ztree2stata sxpose

clear all
set more off

** define working directories
	global path_stata ".../REPLICATION_PACKAGE/stata"
	global path_sessions ".../REPLICATION_PACKAGE/raw data sessions"
	global path_facereader ".../REPLICATION_PACKAGE/raw data FaceReader"

****************
** load subjects table 

* define the number of sessions
global no_session = 14
* define the number of parts in the experiment (ztree treatment files)
global no_treat = 4

* loop over the number of sessions
forvalues i=1(1)$no_session {
	* load the dataset using ZTREE2STATA
	cd "$path_sessions/s`i'"
	ztree2stata subjects using "Dataset.xls", clear
	cd "$path_stata"
	
	* save parts of each session separately
	forvalues k=1(1)$no_treat {
		preserve
		keep if treatment == `k'
		save Session`i'T`k'.dta, replace
		restore
	}
	
	* merge parts of each session
	use Session`i'T1.dta, clear
	forvalues k=2(1)$no_treat {
		merge m:1 Subject using Session`i'T`k'.dta, update
		drop _merge
		append using Session`i'T`k'.dta
	}
	
	* save merged files
	drop session
	gen session = `i'
	save Session`i'.dta, replace
	
	* load questionnaires
	cd "$path_sessions/s`i'"
	import delimited "Quest.sbj", clear
	sxpose, force firstnames destring clear
	drop _var1
	cd "$path_stata"
	save Questionnaire`i'.dta, replace
	
	* merge subject tables and questionnaires
	use Session`i'.dta, clear
	merge m:1 Subject using Questionnaire`i'.dta
	drop _merge
	drop session
	gen session = `i'
	*replace Subject = (`i'*100)+Subject
	save DataS`i'.dta, replace
	
	* delete the rest
	forvalues k=1(1)$no_treat {
		erase Session`i'T`k'.dta
		
	}
		erase Session`i'.dta
		erase Questionnaire`i'.dta
}

cd "$path_stata"

* append all sessions
use DataS1.dta, clear
forvalues i=2(1)$no_session {
append using DataS`i'.dta
}

* delete the rest
forvalues i=1(1)$no_session {
erase DataS`i'.dta
}

* save the dataset
rename treatment part
sort session part Period Subject
save "auctions_14_ind.dta", replace


*******************
***load contracts table into a separate datset
forvalues i=1(1)$no_session {
cd "$path_sessions/s`i'"
	ztree2stata contracts using "Dataset.xls", clear except(explanation TimeWeiterRaven TimeBestaetige TimeWeiterEAR)
	cd "$path_stata"	
	drop session
	gen session = `i'
	save Contracts`i'.dta, replace
	}
	
use Contracts1.dta, clear
	forvalues i=2(1)$no_session {
	append using Contracts`i'.dta
	}
rename treatment part
save "contracts_14.dta", replace
forvalues i=1(1)$no_session {
	erase Contracts`i'.dta
	}

***clean contracts table dataset
clear
use "contracts_14.dta"
gen treatment=1 if session==1 | session==5
replace treatment=2 if session==3 | session==7
replace treatment=3 if session==4 | session==6 | session==9 | session==12 | session==14
replace treatment=4 if session==2 | session==8 | session==10 | session==11 | session==13
label def treat 1 "IND FPA" 2 "IND EA" 3 "TEAM FPA" 4 "TEAM EA"
label values treatment treat
gen individual=1 if treatment<3
replace individual=0 if treatment>=3
label def ind 1 "Individual treatments" 0 "Team treatments"
label values individual ind
label var part "Part of the experiment"
label define parts 1 "Part 1: FPA or EA" 2 "Part 2: SPA" 3 "Part 3: Risk elicitation" 4 "Part 4: Raven's test"
label values part parts
gen auction_type=1 if treatment==1 | treatment==3
replace auction_type=2 if treatment==2 | treatment==4
label def auction 1 "FPA" 2 "EA"
label values auction_type auction
drop if individual==1
drop explanation5- TimeSPAExplanationChatMessage
rename cSubject Subject
rename cTeam Team
rename cTeamID TeamID
rename cRound round
gen unique_id_ind=session*100+Subject
sort session part Period round bid_number
save "contracts_14.dta", replace

***clean subjects table dataset
clear
use auctions_14_ind.dta
gen treatment=1 if session==1 | session==5
replace treatment=2 if session==3 | session==7
replace treatment=3 if session==4 | session==6 | session==9 | session==12 | session==14
replace treatment=4 if session==2 | session==8 | session==10 | session==11 | session==13
label def treat 1 "IND FPA" 2 "IND EA" 3 "TEAM FPA" 4 "TEAM EA"
label values treatment treat
gen individual=1 if treatment<3
replace individual=0 if treatment>=3
label def ind 1 "Individual treatments" 0 "Team treatments"
label values individual ind
drop seat
encode gender, gen(sex)
drop gender
rename sex gender
replace gender=gender-1
label def gender 1"female" 0 "male"
label values gender gender
gen mathgrade=1 if math_grade=="sehr gut"
replace mathgrade=2 if math_grade=="gut"
replace mathgrade=3 if math_grade=="befriedigend"
replace mathgrade=4 if math_grade=="ausreichend"
replace mathgrade=5 if math_grade=="mangelhaft"
drop math_grade
label define grade 1 "sehr gut" 2 "gut" 3 "befriedigend" 4 "ausreichend" 5 "mangelhaft"
label var mathgrade "Math grade in the last school year"
label values mathgrade grade
gen mood1=1 if stimmung1=="sehr gut"
replace mood1=2 if stimmung1=="gut"
replace mood1=3 if stimmung1=="neutral"
replace mood1=4 if stimmung1=="schlecht"
replace mood1=5 if stimmung1=="sehr schlecht"
drop stimmung1
label define mood 1 "sehr gut" 2 "gut" 3 "neutral" 4 "schlecht" 5 "sehr schlecht"
label var mood1 "Mood today"
label values mood1 mood
gen mood2=1 if stimmung2=="sehr gut"
replace mood2=2 if stimmung2=="gut"
replace mood2=3 if stimmung2=="neutral"
replace mood2=4 if stimmung2=="schlecht"
replace mood2=5 if stimmung2=="sehr schlecht"
drop stimmung2
label var mood2 "Mood during the experiment"
label values mood2 mood
gen experience=1 if frequency=="bisher noch nie"
replace experience=2 if frequency=="bisher 1 mal"
replace experience=3 if frequency=="2 - 5 mal"
replace experience=4 if frequency=="öfter"
drop frequency
label define freq 1 "bisher noch nie" 2 "bisher 1 mal" 3 "2 - 5 mal" 4 "öfter"
label var experience "Frequency of participation in experiments"
label values experience freq
label var kennen "The number of aquaintances in the room"
drop tables
drop Group
drop Participate
label var num_correct "Number of correct answers in Raven's test"
drop qnum
drop muC
label var part "Part of the experiment"
label define parts 1 "Part 1: FPA or EA" 2 "Part 2: SPA" 3 "Part 3: Risk elicitation" 4 "Part 4: Raven's test"
label values part parts
replace invest=invest_fin if individual==0
drop invest_fin
label var invest "Amount invested into the risky asset"
rename bid_sp bid_sp_proposal
label var bid_sp_proposal "Individual proposal for bid in SPA"
label var bid_sp_fin "Final bid in SPA"
rename bid_fp bid_fp_proposal
label var bid_fp_proposal "Individual proposal for bid in FPA"
label var bid_fp_fin "Final bid in FPA"
replace confirm=0 if confirm==2
label var confirm "Indicator: individual confirmed his bid proposal"
label def conf 1 "bid proposal confirmed" 0 "bid proposal changed"
label values confirm conf
drop LeaveStage
label var winner "Indicator: winning an auction"
label def win 1 "won the auction" 0 "did not win the auction"
label values winner win
label var NetProfit "Net profit from the auction" 
drop RankMarket
drop click
drop randwin
drop RandomID
drop Rank
drop winner1
drop winner2
drop winrandwin
drop RandomTeam
drop RankTeam
drop Fixed
drop client
drop last_bid
drop last_invest
drop last_ans
gen auction_type=1 if treatment==1 | treatment==3
replace auction_type=2 if treatment==2 | treatment==4
label def auction 1 "FPA" 2 "EA"
label values auction_type auction
gen MGroup=session*100+MatchingGroup if part<3
replace MGroup=session*100 if MGroup==. & part<3
drop MatchingGroup
gen unique_id_ind=session*100+Subject

egen stay_ea=rowtotal(active*) if auction_type==2 & part==1 //number of active rounds in EA 
gen delta_bid=value-bid_fp_fin if auction_type==1 //bid shading in FPA and EA
egen maxbid_ea =rowmax( bid_ea_fin* )
replace delta_bid=value-maxbid_ea if auction_type==2
label var delta_bid "Value minus (maximum) bid"
gsort unique_id_ind -part Period
gen delta_sp_dir=bid_sp_fin-value if part==2 //deviation from rational strategy in SPA
replace delta_sp_dir=delta_sp_dir[_n-1] if unique_id_ind==unique_id_ind[_n-1] & part<2
gen delta_sp=abs(delta_sp_dir) // absolute deviation from rational strategy in SPA
gen all_stay_ea=0 //number of rounds when all 3 bidders on the market are active
forvalues i=1(1)31 {
replace all_stay_ea=all_stay_ea+1 if marketsize`i'==3
}
replace all_stay_ea=. if auction_type==1
replace all_stay_ea=. if part>1
gsort -individual session part Period Market -value Team //efficiency indicator
gen highest_value=1 in 1
replace highest_value=1 if individual==1 & part<3 & Market!=Market[_n-1]
replace highest_value=1 if individual==1 & part<3 & Market==Market[_n-1] & value==value[_n-1] & highest_value[_n-1]==1
replace highest_value=0 if individual==1 & part<3 & highest_value==.
gsort individual session part Period Market -value Team
replace highest_value=1 in 1
replace highest_value=1 if individual==0 & part<3 & Market!=Market[_n-1]
replace highest_value=1 if individual==0 & part<3 & Market==Market[_n-1] & Team!=Team[_n-1] & value==value[_n-3] & highest_value[_n-3]==1
replace highest_value=0 if individual==0 & part<3 & Market==Market[_n-1] & Team!=Team[_n-1] & highest_value==.
replace highest_value=highest_value[_n-1] if individual==0 & part<3 & Market==Market[_n-1] & Team==Team[_n-1]
gsort individual session part Period Market -winner Team
gen efficiency=1 if highest_value==1 & winner==1 & part<3
replace efficiency=0 if highest_value==0 & winner==1 & part<3
replace efficiency=efficiency[_n-1] if part<3 & Market==Market[_n-1]
replace efficiency=0 if efficiency ==. & part<3
*relative bids in EA
forvalues i=1/30 {
replace bid_ea_fin`i'=. if bid_ea_fin`i'<=0
}
forvalues i=1/30 {
gen rbid_ea`i'=bid_ea_fin`i'/min_bid`i'
}
egen mean_rbid_ea=rowmean(rbid_ea*)
drop rbid_ea*
sort session part Period Subject

gen maxmathgrade=mathgrade if individual==1 &part==1
gsort individual session part Team mathgrade
replace maxmathgrade=mathgrade in 1
replace maxmathgrade=mathgrade if Team!=Team[_n-1] & part==1 & individual==0
replace maxmathgrade=maxmathgrade[_n-1] if Team==Team[_n-1] & part==1 & individual==0

save auctions_14_ind, replace
drop if part>1
*Add 2nd and 3rd highest value
bysort session Period Market (value) : gen max_value = value[_N]
bysort session Period Market (value) : gen second_value = value[_N-1] if individual==1
bysort session Period Market (value) : replace second_value = value[_N-3] if individual==0
bysort session Period Market (value) : gen third_value = value[_N-2] if individual==1
bysort session Period Market (value) : replace third_value = value[_N-6] if individual==0
gen delta_value12 = max_value - second_value
gen delta_value12_perc = (max_value / second_value) - 1
gen delta_value13 = max_value - third_value
gen delta_value13_perc = (max_value / third_value) - 1
gen delta_value23 = second_value - third_value
gen delta_value23_perc = (second_value / third_value) - 1
merge 1:1 part Period unique_id_ind using auctions_14_ind, nogenerate
*Add relative profit
gen rel_NetProfit = NetProfit / value

sort session part Period Subject
save auctions_14_ind, replace


***merge contracts with subjects
clear
use contracts_14.dta
keep if auction_type==1 & part==1
keep part Period bid_number bid_fp unique_id_ind
keep if bid_number==1
rename bid_fp bid_fp_proposal
drop bid_number
merge 1:1 part Period unique_id_ind using auctions_14_ind
drop _merge
save auctions_14_ind, replace
*erase contracts_14.dta


***merge with FaceReader data
clear
use "$path_facereader/facereader_valence1214.dta"
merge 1:1 Session Subject Period using "$path_facereader/facereader_data.dta"
gen part=1
rename Session session
sort session part Period Subject
drop _merge
merge 1:1 session part Period Subject using auctions_14_ind
sort session part Period Subject
drop _merge
save auctions_14_ind.dta, replace


******** collapse data on team level (for graphs and regressions)
clear
use auctions_14_ind
gen unique_id=session*100+Subject if individual==1
replace unique_id=session*100+Team if individual==0
collapse Valence Neutral Happy Sad Angry Surprised Scared Disgusted emo_numobs Profit TotalProfit value Market bid_fp_fin bid_fp_proposal winner price NetProfit SelectedPeriod bid_sp_fin invest return answer* num_correct min_bid* cprice* bid_ea_fin* active* marketsize* agree highest_bid treatment individual MGroup auction_type stay_ea delta_bid maxbid_ea delta_sp_dir delta_sp all_stay_ea highest_value efficiency gender mathgrade maxmathgrade mood1 mood2 experience mean_rbid_ea max_value second_value third_value delta_value12 delta_value12_perc delta_value13 delta_value13_perc delta_value23 delta_value23_perc rel_NetProfit, by (session part Period unique_id)
sort session part Period unique_id
gen risk_par_r=log(1.25)/log(2.5*invest) if part==1 & invest>0
replace risk_par_r=1 if part==1 & invest==0
label var risk_par_r "Risk aversion CRRA parameter"
gen bid_fp_predict=2/(3-risk_par_r)*value if part==1 & auction_type==1
label var bid_fp_predict "Estimated optimal bid in FPA with risk averse bidders"
gen delta_bid_predict=value-bid_fp_predict if part==1 & auction_type==1
label var bid_fp_predict "Estimated optimal bid shading in FPA with risk averse bidders"
gen bid_fp_diff=bid_fp_fin-bid_fp_predict if part==1 & auction_type==1
label var bid_fp_diff "Deviation from optimal bid shading in FPA with risk averse bidders"
gen delta_bid_proposal=value-bid_fp_proposal if auction_type==1 & part==1
*generate means for graphs
by treatment part Period, sort: egen MeanProfit=mean(NetProfit)
gen MProfit_ind_fpa=MeanProfit if treatment==1
gen MProfit_team_fpa=MeanProfit if treatment==3
gen MProfit_ind_ea=MeanProfit if treatment==2
gen MProfit_team_ea=MeanProfit if treatment==4
by treatment part Period, sort: egen MeanShad=mean(delta_bid)
gen MShad_ind_fpa=MeanShad if treatment==1
gen MShad_team_fpa=MeanShad if treatment==3
gen MShad_ind_ea=MeanShad if treatment==2
gen MShad_team_ea=MeanShad if treatment==4
by treatment part Period, sort: egen MeanPrice=mean(price)
gen MPrice_ind_fpa=MeanPrice if treatment==1
gen MPrice_team_fpa=MeanPrice if treatment==3
gen MPrice_ind_ea=MeanPrice if treatment==2
gen MPrice_team_ea=MeanPrice if treatment==4
by treatment part Period, sort: egen MeanEff=mean(efficiency)
gen MEff_ind_fpa=MeanEff if treatment==1
gen MEff_team_fpa=MeanEff if treatment==3
gen MEff_ind_ea=MeanEff if treatment==2
gen MEff_team_ea=MeanEff if treatment==4
by treatment part Period, sort: egen MeanStay=mean(stay_ea)
gen MStay_ind_ea=MeanStay if treatment==2
gen MStay_team_ea=MeanStay if treatment==4
by treatment part Period, sort: egen MeanRbid=mean(mean_rbid_ea)
gen MRbid_ind_ea=MeanRbid if treatment==2
gen MRbid_team_ea=MeanRbid if treatment==4
by treatment part Period, sort: egen MeanShadPred=mean(delta_bid_predict)
gen MShadPred_ind_fpa=MeanShadPred if treatment==1
gen MShadPred_team_fpa=MeanShadPred if treatment==3
by treatment part Period, sort: egen MeanShadProp=mean(delta_bid_proposal)
gen MShadProp_ind_fpa=MeanShadProp if treatment==1
gen MShadProp_team_fpa=MeanShadProp if treatment==3

by treatment part Period, sort: egen MeanShadPredNozero=mean(delta_bid_predict) if invest>0
gen MShadPredNozero_ind_fpa=MeanShadPredNozero if treatment==1
gen MShadPredNozero_team_fpa=MeanShadPredNozero if treatment==3
by treatment part Period, sort: egen MeanShadNozero=mean(delta_bid) if invest>0
gen MShadNozero_ind_fpa=MeanShadNozero if treatment==1
gen MShadNozero_team_fpa=MeanShadNozero if treatment==3

by treatment part unique_id, sort: egen MeanShadid=mean(delta_bid)
gen MShadid_ind_fpa=MeanShadid if treatment==1
gen MShadid_team_fpa=MeanShadid if treatment==3

save auctions_14_team, replace


**** collapse data on matching group level (for non-parametric tests)
clear
use auctions_14_team
sort Market
egen highestvalinmarket = max(value), by(Market)
collapse Valence Neutral Happy Sad Angry Surprised Scared Disgusted emo_numobs Profit TotalProfit value bid_fp_fin winner price NetProfit SelectedPeriod bid_sp_fin invest return answer* num_correct min_bid* cprice* bid_ea_fin* active* marketsize* agree highest_bid treatment individual auction_type stay_ea delta_bid maxbid_ea delta_sp_dir delta_sp all_stay_ea highest_value efficiency gender mathgrade mood1 mood2 experience mean_rbid_ea bid_fp_diff risk_par_r if part==1, by (session MGroup)
save auctions_14_mg, replace


**** collapse data on market level (for regressions)
use auctions_14_team
collapse Valence Neutral Happy Sad Angry Surprised Scared Disgusted emo_numobs Profit TotalProfit value bid_fp_fin winner price NetProfit SelectedPeriod bid_sp_fin invest return answer* num_correct min_bid* cprice* bid_ea_fin* active* marketsize* agree highest_bid treatment individual MGroup auction_type stay_ea delta_bid maxbid_ea delta_sp_dir delta_sp all_stay_ea highest_value efficiency gender mathgrade mood1 mood2 experience mean_rbid_ea max_value second_value third_value delta_value12 delta_value12_perc delta_value13 delta_value13_perc delta_value23 delta_value23_perc, by (session part Period Market)
gen unique_id_mrkt=session*100+Market
save auctions_14_market, replace
